from django.db import models 
from models import *
# from inventory_app.models import *
# from sales_app.models import *


 
class Department(models.Model):
    name = models.CharField(max_length=100, unique=True)
    rank = models.IntegerField() 
    status = models.BooleanField(default=True)
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'department_list'

    def __str__(self):
        return self.name


class Designation(models.Model):
    name = models.CharField(max_length=100, unique=True)
    rank = models.IntegerField() 
    status = models.BooleanField(default=True)
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'designation_list'

    def __str__(self):
        return self.name



class Module(models.Model):
    name = models.CharField(max_length=100, unique=True)
    rank = models.IntegerField(default=1)
    icon = models.CharField(max_length=100, blank=True, null=True)
    color = models.CharField(max_length=100, blank=True, null=True)
    status = models.BooleanField(default=True)
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'module_list'

    def __str__(self):
        return self.name

class ModuleMenu(models.Model):
    module = models.ForeignKey(Module, on_delete=models.SET_NULL, null=True, related_name='menus')
    name = models.CharField(max_length=100, unique=True)
    rank = models.IntegerField(default=1)
    menu_url = models.CharField(max_length=100, blank=True, null=True)
    icon = models.CharField(max_length=100, blank=True, null=True)
    color = models.CharField(max_length=100, blank=True, null=True)
    status = models.BooleanField(default=True)
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'module_menu_list'

    def __str__(self):
        return self.name

 

class Employee(models.Model):
    employee_id = models.CharField(max_length=20, unique=True)
    password = models.CharField(max_length=128)
    full_name = models.CharField(max_length=150)
    email = models.EmailField(unique=True)
    phone = models.CharField(max_length=20, blank=True, null=True)
    address = models.TextField(blank=True, null=True)
    department = models.ForeignKey(Department, on_delete=models.SET_NULL, null=True, related_name='employees')
    designation = models.ForeignKey(Designation, on_delete=models.SET_NULL, null=True, related_name='employees')
    join_date = models.DateField()
    salary_basic = models.DecimalField(max_digits=10, decimal_places=2)
    salary_gross = models.DecimalField(max_digits=10, decimal_places=2)
    national_id = models.CharField(max_length=30, blank=True, null=True)
    photo = models.ImageField(upload_to='employee_photos/', blank=True, null=True)
    is_active = models.BooleanField(default=True)
    is_delete = models.BooleanField(default=False)
    created = models.DateTimeField(auto_now_add=True)
    updated = models.DateTimeField(auto_now=True)

    class Meta:
        db_table = 'employee_list'
        ordering = ['-created_at']

    def __str__(self):
        return f"{self.employee_id} - {self.full_name}"
    

class UserAccessControl(models.Model):
    module = models.ForeignKey(Module, on_delete=models.SET_NULL, null=True, related_name='access_controls')
    menu = models.ForeignKey(ModuleMenu, on_delete=models.SET_NULL, null=True, related_name='access_controls')
    employee = models.ForeignKey(Employee, on_delete=models.SET_NULL, null=True, related_name='accesses')
    created_by = models.ForeignKey(Employee, on_delete=models.SET_NULL, null=True, related_name='created_accesses')

    can_create = models.BooleanField(default=False)
    can_read = models.BooleanField(default=False)
    can_update = models.BooleanField(default=False)
    can_delete = models.BooleanField(default=False)

    class Meta:
        db_table = 'user_access_control'

    def __str__(self):
        return f"AccessControl for {self.employee} on {self.menu}"


class Attendance(models.Model):
    employee = models.ForeignKey(Employee, on_delete=models.CASCADE)
    date = models.DateField()
    status = models.CharField(max_length=10, choices=[('Present', 'Present'), ('Absent', 'Absent')])

    class Meta:
        db_table = 'attendance'

class Payroll(models.Model):
    employee = models.ForeignKey(Employee, on_delete=models.CASCADE)
    month = models.CharField(max_length=20)
    salary = models.DecimalField(max_digits=10, decimal_places=2)
    paid_on = models.DateField()

    class Meta:
        db_table = 'payroll'

class LeaveRequest(models.Model):
    employee = models.ForeignKey(Employee, on_delete=models.CASCADE)
    from_date = models.DateField()
    to_date = models.DateField()
    reason = models.TextField()
    status = models.CharField(max_length=20, choices=[('Pending', 'Pending'), ('Approved', 'Approved'), ('Rejected', 'Rejected')], default='Pending')

    class Meta:
        db_table = 'leave_request'

 

# 4. Delivery / Shipping Management
class ShippingMethod(models.Model):
    name = models.CharField(max_length=100)
    cost = models.DecimalField(max_digits=10, decimal_places=2)
    estimated_time = models.CharField(max_length=100)

    class Meta:
        db_table = 'shipping_method'

class ShippingAddress(models.Model):
    customer = models.ForeignKey('Customer', on_delete=models.CASCADE)
    address_line = models.TextField()
    city = models.CharField(max_length=100)
    postal_code = models.CharField(max_length=20)
    country = models.CharField(max_length=100)

    class Meta:
        db_table = 'shipping_address'

class DeliveryStatus(models.Model):
    order = models.ForeignKey('SalesOrder', on_delete=models.CASCADE)
    status = models.CharField(max_length=50)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        db_table = 'delivery_status'

class CourierService(models.Model):
    name = models.CharField(max_length=100)
    contact_number = models.CharField(max_length=20)

    class Meta:
        db_table = 'courier_service'

class TrackingInfo(models.Model):
    order = models.ForeignKey('SalesOrder', on_delete=models.CASCADE)
    tracking_number = models.CharField(max_length=100)
    courier = models.ForeignKey(CourierService, on_delete=models.SET_NULL, null=True)
    status = models.CharField(max_length=100)
    updated_on = models.DateTimeField(auto_now=True)

    class Meta:
        db_table = 'tracking_info'

# 5. Payment & Accounting
class PaymentMethod(models.Model):
    name = models.CharField(max_length=50)
    details = models.TextField(blank=True)

    class Meta:
        db_table = 'payment_method'

class PaymentTransaction(models.Model):
    order = models.ForeignKey('SalesOrder', on_delete=models.CASCADE)
    method = models.ForeignKey(PaymentMethod, on_delete=models.SET_NULL, null=True)
    amount = models.DecimalField(max_digits=12, decimal_places=2)
    date = models.DateField()

    class Meta:
        db_table = 'payment_transaction'

class ExpenseCategory(models.Model):
    name = models.CharField(max_length=100)

    class Meta:
        db_table = 'expense_category'

class Expense(models.Model):
    category = models.ForeignKey(ExpenseCategory, on_delete=models.CASCADE)
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    date = models.DateField()
    notes = models.TextField(blank=True)

    class Meta:
        db_table = 'expense'

class Income(models.Model):
    source = models.CharField(max_length=100)
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    date = models.DateField()
    notes = models.TextField(blank=True)

    class Meta:
        db_table = 'income'

class ChartOfAccounts(models.Model):
    name = models.CharField(max_length=100)
    code = models.CharField(max_length=20)

    class Meta:
        db_table = 'chart_of_accounts'

class JournalEntry(models.Model):
    account = models.ForeignKey(ChartOfAccounts, on_delete=models.CASCADE)
    debit = models.DecimalField(max_digits=12, decimal_places=2, default=0)
    credit = models.DecimalField(max_digits=12, decimal_places=2, default=0)
    date = models.DateField()

    class Meta:
        db_table = 'journal_entry'

class LedgerEntry(models.Model):
    account = models.ForeignKey(ChartOfAccounts, on_delete=models.CASCADE)
    date = models.DateField()
    amount = models.DecimalField(max_digits=12, decimal_places=2)
    type = models.CharField(max_length=10, choices=[('Debit', 'Debit'), ('Credit', 'Credit')])

    class Meta:
        db_table = 'ledger_entry'

class BankAccount(models.Model):
    bank_name = models.CharField(max_length=100)
    account_number = models.CharField(max_length=50)
    balance = models.DecimalField(max_digits=15, decimal_places=2)

    class Meta:
        db_table = 'bank_account'

class CashRegister(models.Model):
    name = models.CharField(max_length=100)
    balance = models.DecimalField(max_digits=12, decimal_places=2)

    class Meta:
        db_table = 'cash_register'

# 8. Settings & Configuration
class CompanyInfo(models.Model):
    name = models.CharField(max_length=255)
    address = models.TextField()
    phone = models.CharField(max_length=20)
    email = models.EmailField()

    class Meta:
        db_table = 'company_info'

class TaxSettings(models.Model):
    tax_name = models.CharField(max_length=50)
    rate = models.DecimalField(max_digits=5, decimal_places=2)

    class Meta:
        db_table = 'tax_settings'

class Currency(models.Model):
    code = models.CharField(max_length=10)
    symbol = models.CharField(max_length=10)
    name = models.CharField(max_length=50)

    class Meta:
        db_table = 'currency'

class SystemSettings(models.Model):
    key = models.CharField(max_length=100, unique=True)
    value = models.TextField()

    class Meta:
        db_table = 'system_settings'

class Role(models.Model):
    name = models.CharField(max_length=50)

    class Meta:
        db_table = 'role'

# 9. Customer Support & Feedback
class ContactRequest(models.Model):
    name = models.CharField(max_length=100)
    email = models.EmailField()
    subject = models.CharField(max_length=200)
    message = models.TextField()
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'contact_request'

class ProductReview(models.Model):
    product = models.ForeignKey(Product, on_delete=models.CASCADE)
    customer = models.ForeignKey('Customer', on_delete=models.CASCADE)
    rating = models.IntegerField()
    review = models.TextField()
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'product_review'

class SupportTicket(models.Model):
    customer = models.ForeignKey('Customer', on_delete=models.CASCADE)
    subject = models.CharField(max_length=200)
    message = models.TextField()
    status = models.CharField(max_length=50, choices=[('Open', 'Open'), ('Closed', 'Closed')], default='Open')
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'support_ticket'

# 10. Marketing
class EmailCampaign(models.Model):
    subject = models.CharField(max_length=200)
    content = models.TextField()
    sent_on = models.DateTimeField(null=True, blank=True)

    class Meta:
        db_table = 'email_campaign'

class SMSCampaign(models.Model):
    message = models.TextField()
    sent_on = models.DateTimeField(null=True, blank=True)

    class Meta:
        db_table = 'sms_campaign'

class Subscriber(models.Model):
    email = models.EmailField(unique=True)
    subscribed_on = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'subscriber'

class PromotionBanner(models.Model):
    title = models.CharField(max_length=200)
    image = models.ImageField(upload_to='promotion_banners/')
    link = models.URLField(blank=True, null=True)

    class Meta:
        db_table = 'promotion_banner'

class SocialMediaPost(models.Model):
    platform = models.CharField(max_length=100)
    content = models.TextField()
    posted_on = models.DateTimeField()

    class Meta:
        db_table = 'social_media_post'

# 11. Website CMS
class Page(models.Model):
    title = models.CharField(max_length=200)
    slug = models.SlugField(unique=True)
    content = models.TextField()

    class Meta:
        db_table = 'page'

class Menu(models.Model):
    name = models.CharField(max_length=100)
    link = models.CharField(max_length=200)

    class Meta:
        db_table = 'menu'

class Slider(models.Model):
    title = models.CharField(max_length=100)
    image = models.ImageField(upload_to='sliders/')
    link = models.URLField(blank=True, null=True)

    class Meta:
        db_table = 'slider'

class BlogPost(models.Model):
    title = models.CharField(max_length=200)
    slug = models.SlugField(unique=True)
    content = models.TextField()
    published_on = models.DateTimeField(auto_now_add=True)

    class Meta:
        db_table = 'blog_post'

class FAQ(models.Model):
    question = models.CharField(max_length=300)
    answer = models.TextField()

    class Meta:
        db_table = 'faq'


